import { useEffect, useMemo, useRef, useState } from "react";
import {
  AudioFormat,
  CommitStrategy,
  RealtimeEvents,
  Scribe,
} from "@elevenlabs/client";
import "./App.css";

const SERVER_ENDPOINT = `${
  import.meta.env.VITE_SERVER_ORIGIN ?? "http://localhost:3001"
}/scribe-token`;

function App() {
  const [token, setToken] = useState(null);
  const [connected, setConnected] = useState(false);
  const [statusMessage, setStatusMessage] = useState("初期化中…");
  const [connectionPhase, setConnectionPhase] = useState("initializing");
  const [partialText, setPartialText] = useState("");
  const [committedText, setCommittedText] = useState("");
  const [errorMessage, setErrorMessage] = useState(null);
  const connectionRef = useRef(null);

  const isConnecting = useMemo(
    () =>
      ["initializing", "fetching_token", "connecting"].includes(
        connectionPhase
      ),
    [connectionPhase]
  );

  const fetchToken = async () => {
    try {
      setStatusMessage("トークン取得中…");
      setConnectionPhase("fetching_token");
      setErrorMessage(null);

      const response = await fetch(SERVER_ENDPOINT);

      if (!response.ok) {
        throw new Error(`トークン取得に失敗しました (${response.status})`);
      }

      const data = await response.json();

      if (!data?.token) {
        throw new Error("レスポンスにトークンが含まれていません。");
      }

      setToken(data.token);
      setStatusMessage("接続準備中…");
      setConnectionPhase("connecting");
    } catch (error) {
      console.error("Failed to fetch token:", error);
      setStatusMessage("トークン取得エラー");
      setConnectionPhase("error");
      setErrorMessage(error.message);
    }
  };

  useEffect(() => {
    fetchToken();
    return () => {
      if (connectionRef.current) {
        connectionRef.current.close();
        connectionRef.current = null;
      }
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  useEffect(() => {
    if (!token) return undefined;

    let mounted = true;
    let unsubscribe = null;

    (async () => {
      try {
        setStatusMessage("接続試行中…");
        setConnectionPhase("connecting");

        const connection = await Scribe.connect({
          token,
          modelId: "scribe_v2_realtime",
          languageCode: "ja",
          audioFormat: AudioFormat.PCM_16000,
          commitStrategy: CommitStrategy.VAD,
          vadSilenceThresholdSecs: 1.5,
          vadThreshold: 0.4,
          minSpeechDurationMs: 100,
          minSilenceDurationMs: 100,
          microphone: {
            echoCancellation: true,
            noiseSuppression: true,
            autoGainControl: true,
          },
        });

        if (!mounted) {
          connection.close();
          return;
        }

        connectionRef.current = connection;
        setConnected(true);
        setStatusMessage("接続中");
        setConnectionPhase("connected");

        const handlePartial = (message) => {
          setPartialText(message?.text ?? "");
        };

        const handleCommitted = (message) => {
          if (message?.text) {
            setCommittedText((prev) =>
              prev ? `${prev}\n${message.text}` : message.text
            );
          }
          setPartialText("");
        };

        const handleCommittedWithTimestamps = (message) => {
          if (message?.text) {
            setCommittedText((prev) =>
              prev ? `${prev}\n${message.text}` : message.text
            );
          }
          setPartialText("");
        };

        const handleSessionStart = () => {
          setStatusMessage("セッション開始");
        };

        const handleClose = () => {
          setConnected(false);
          setStatusMessage("切断済み");
          setConnectionPhase("disconnected");
          setToken(null);
          setPartialText("");
          connectionRef.current = null;
        };

        const handleError = (error) => {
          console.error("Realtime error:", error);
          setErrorMessage(
            error?.error ?? "リアルタイム接続でエラーが発生しました。"
          );
          setConnected(false);
          setStatusMessage("エラーが発生しました");
          setConnectionPhase("error");
        };

        const handleAuthError = (error) => {
          console.error("Auth error:", error);
          setErrorMessage("認証エラーが発生しました。再接続してください。");
          setConnected(false);
          setStatusMessage("認証エラー");
          setConnectionPhase("error");
        };

        connection.on(RealtimeEvents.SESSION_STARTED, handleSessionStart);
        connection.on(RealtimeEvents.PARTIAL_TRANSCRIPT, handlePartial);
        connection.on(RealtimeEvents.COMMITTED_TRANSCRIPT, handleCommitted);
        connection.on(
          RealtimeEvents.COMMITTED_TRANSCRIPT_WITH_TIMESTAMPS,
          handleCommittedWithTimestamps
        );
        connection.on(RealtimeEvents.ERROR, handleError);
        connection.on(RealtimeEvents.AUTH_ERROR, handleAuthError);
        connection.on(RealtimeEvents.CLOSE, handleClose);

        unsubscribe = () => {
          connection.off(RealtimeEvents.SESSION_STARTED, handleSessionStart);
          connection.off(RealtimeEvents.PARTIAL_TRANSCRIPT, handlePartial);
          connection.off(
            RealtimeEvents.COMMITTED_TRANSCRIPT,
            handleCommitted
          );
          connection.off(
            RealtimeEvents.COMMITTED_TRANSCRIPT_WITH_TIMESTAMPS,
            handleCommittedWithTimestamps
          );
          connection.off(RealtimeEvents.ERROR, handleError);
          connection.off(RealtimeEvents.AUTH_ERROR, handleAuthError);
          connection.off(RealtimeEvents.CLOSE, handleClose);
        };
      } catch (error) {
        console.error("Failed to connect:", error);
        setStatusMessage("接続失敗");
        setConnectionPhase("error");
        setErrorMessage(error.message ?? "接続処理でエラーが発生しました。");
        setConnected(false);
      }
    })();

    return () => {
      mounted = false;
      if (unsubscribe) {
        unsubscribe();
      }
      if (connectionRef.current) {
        connectionRef.current.close();
        connectionRef.current = null;
      }
    };
  }, [token]);

  const handleDisconnect = () => {
    if (connectionRef.current) {
      connectionRef.current.close();
      connectionRef.current = null;
    }
    setConnected(false);
    setStatusMessage("切断済み");
    setConnectionPhase("disconnected");
    setToken(null);
    setPartialText("");
    setErrorMessage(null);
  };

  const handleReconnect = async () => {
    setCommittedText("");
    setConnectionPhase("fetching_token");
    await fetchToken();
  };

  return (
    <div className="app">
      <header className="app__header">
        <h1>Scribe v2 Realtime デモ</h1>
        <p className="app__status">
          状態:{" "}
          <span
            className={`status status--${
              connected ? "connected" : isConnecting ? "connecting" : "idle"
            }`}
          >
            {connected
              ? "接続中（マイクからリアルタイム文字起こし）"
              : statusMessage}
          </span>
        </p>
        <div className="app__actions">
          <button
            type="button"
            onClick={handleReconnect}
            disabled={isConnecting || connected}
          >
            再接続
          </button>
          <button
            type="button"
            onClick={handleDisconnect}
            disabled={!connected}
            className="button--danger"
          >
            切断
          </button>
        </div>
        {errorMessage ? (
          <p className="app__error">エラー: {errorMessage}</p>
        ) : null}
      </header>

      <main className="app__content">
        <section>
          <h2>部分文字起こし（リアルタイム更新）</h2>
          <pre className="transcript transcript--partial">
            {partialText || "ここにリアルタイムの途中経過が表示されます"}
          </pre>
        </section>
        <section>
          <h2>コミット済みテキスト</h2>
          <pre className="transcript transcript--committed">
            {committedText || "ここに確定した文字起こしが溜まっていきます"}
          </pre>
        </section>
      </main>
    </div>
  );
}

export default App;
